# objective.py
from typing import Any, Dict, Tuple

class Objective:
    def __init__(self, evaluator, bench: str = "darts-xgb-v1.0", metric: str = "auto"):
        self.evaluator = evaluator
        self.bench = bench.lower()
        self.metric = metric  # "auto" | "error" | "acc"
        self.sim_time = 0.0

    def _as_loss(self, y: float) -> float:
        y = float(y)
        # Auto detection: DARTS usually uses "accuracy", FBNet usually uses "error"
        if self.metric == "auto":
            if "darts" in self.bench:
                acc = y / 100.0 if y > 1.5 else y
                return 1.0 - acc                      # → error (0~1), minimize
            else:
                # Fallback: >1.5 is considered percentage accuracy
                return (100.0 - y) / 100.0 if y > 1.5 else y
        elif self.metric == "acc":
            acc = y / 100.0 if y > 1.5 else y
            return 1.0 - acc
        elif self.metric == "error":
            return y / 100.0 if y > 1.5 else y
        else:
            return y

    def evaluate(self, cfg_dict: Dict[str, Any]) -> Tuple[float, float]:
        score, rt = self.evaluator.predict(cfg_dict, representation="configspace")
        loss = self._as_loss(score)
        self.sim_time = float(rt)
        return float(loss), float(self.sim_time)
